#!/usr/bin/env raku

# Make sure we get the most recent version
use lib $*PROGRAM.parent;
use Raku::Elements;

# This script reads the files in the resources directory, and updates
# the doc/Raku-Elements.rakudoc file with it

my $groups-io := $*PROGRAM.sibling("resources").add("groups");

my $generator := $*PROGRAM-NAME;
my $generated := DateTime.now.gist.subst(/\.\d+/,'');
my $start     := "=comment PLEASE DON'T CHANGE ANYTHING BELOW THIS LINE";
my $end       := "=comment PLEASE DON'T CHANGE ANYTHING ABOVE THIS LINE";

# Slurp the whole file and set up writing to it
my $file := $*PROGRAM.sibling("doc").add("Raku-Elements.rakudoc");
my @lines = $file.lines;  # must be greedy
$*OUT = $file.open(:w);

my sub backtick($name) {
    $name.starts-with("«" | "»") ?? "C<$name>" !! "C«$name»"
}
my sub sanitize($text) {
    $text
      .subst(/ '$*' \w+ /, { "C<$/>" }, :g)
}

# For all the lines in the doc that don't need special handling
while @lines {
    my $line := @lines.shift;

    # nothing to do yet
    unless $line.starts-with($start) {
        say $line;
        next;
    }

    say $start;
    say "=comment Generated on $generated by $generator\n";

    # Skip the old version of the doc
    while @lines {
        last if @lines.shift.starts-with($end);
    }

    my $elements := Raku::Elements.new;
    my @tags = $elements.tags.sort(*.key).map(*.value);
    say "=head2 Raku Element Tags\n";

    for @tags.map(*.name) -> $name {
        say "L<C<$name>|#$name>";
    }

    for @tags {
        say "\n=head3 $_.name()\n\n$_.description()";
    }

    my @groups = $elements.groups.sort(*.key).map(*.value);
    say "\n=head2 Raku Element Groups\n";

    for @groups.map(*.name) -> $name {
        say "L<C<$name>|#$name>";
    }

    for @groups -> $group {
        my $group-name = $group.name;
        say "\n=head2 $group-name";
        say "\nThe $group-name group $_" with $group.description;

        for $group.elements {
            my $head = "&backtick(.name) &sanitize(.tagline)";
            $head = "L<$head|$_.url()>" if .url;
            say "\n=head3 $head";
            if .alternates -> @alternates {
                say "=item alternates: @alternates.map(&backtick)";
            }
            say "=item $_.tags.map({ "L<$_|#$_>" })";
            say "\n$_.description()" if .description;
        }
    }

    # we're done
    say "\n$end";
}

# Close the file properly
$*OUT.close;

# Make sure the README is updated accordingly
run <mi6 build>;

# vim: expandtab shiftwidth=4
