use v6.d;

#| Streamable HTTP transport implementation
unit module MCP::Transport::StreamableHTTP;

=begin pod
=head1 NAME

MCP::Transport::StreamableHTTP - Streamable HTTP transport implementation

=head1 DESCRIPTION

Implements the MCP Streamable HTTP transport using Cro::HTTP. Supports
client-side POST/GET with SSE, and server-side handling of POST requests
plus optional server-initiated SSE streams.

=head1 CLASSES

=head2 StreamableHTTPServerTransport

Server-side transport that listens on an HTTP endpoint and dispatches
JSON-RPC messages received via POST. Supports session management and
SSE streaming for server-initiated messages.

    my $transport = StreamableHTTPServerTransport.new(
        host => 'localhost', port => 8080, path => '/mcp');

Attributes:

=item C<host> — Bind address (default: C<localhost>).
=item C<port> — Bind port (default: C<8080>).
=item C<path> — URL path for the MCP endpoint (default: C</mcp>).

=head2 StreamableHTTPClientTransport

Client-side transport that connects to an MCP HTTP server. Sends
JSON-RPC messages via POST and receives responses and server-initiated
messages via SSE.

    my $transport = StreamableHTTPClientTransport.new(
        url => 'http://localhost:8080/mcp');

Attributes:

=item C<url> — Server endpoint URL.
=item C<headers> — Optional additional HTTP headers (e.g., for authentication).

Key methods:

=item C<terminate-session(--> Promise)> — Send DELETE to terminate the session.

=end pod

use MCP::JSONRPC;
use MCP::Transport::Base;
need MCP::Types;
use JSON::Fast;
use MCP::OAuth;
# Cro::HTTP is loaded dynamically to keep this transport optional.

my constant DEFAULT_PROTOCOL_FALLBACK = '2025-11-25';
my constant DEFAULT_ACCEPT_POST = 'application/json, text/event-stream';
my constant DEFAULT_ACCEPT_SSE = 'text/event-stream';

class X::MCP::Transport::StreamableHTTP is Exception {
    has Str $.message is required;
    method message(--> Str) { $!message } # UNCOVERABLE
}

class X::MCP::Transport::StreamableHTTP::Protocol is X::MCP::Transport::StreamableHTTP {
    method message(--> Str) { "Protocol error: {callsame}" } # UNCOVERABLE
}

class X::MCP::Transport::StreamableHTTP::HTTP is X::MCP::Transport::StreamableHTTP {
    method message(--> Str) { "HTTP error: {callsame}" } # UNCOVERABLE
}

class MCP::Transport::StreamableHTTP::Stream {
    has Str $.id is required;
    has Supplier $.supplier is required;
    has Int $!seq = 0;
    has Int $.history-size = 200;
    has @!history; # [{ id => Str, data => Str }]

    method emit-priming() {
        $!seq++;
        my $id = "{$!id}:{$!seq}";
        my $payload = "id: $id\ndata:\n\n";
        self!store($id, $payload);
        $!supplier.emit($payload);
        $id
    }

    method emit-message(Str $json) {
        $!seq++;
        my $id = "{$!id}:{$!seq}";
        my $payload = self!sse-payload($id, $json);
        self!store($id, $payload);
        $!supplier.emit($payload);
        $id
    }

    method replay-from(Int $seq) {
        for @!history -> %event {
            my ($sid, $sseq) = %event<id>.split(':', 2);
            next unless $sid eq $!id;
            next unless $sseq.Int > $seq;
            $!supplier.emit(%event<data>);
        }
    }

    method !sse-payload(Str $id, Str $json --> Str) {
        my @lines = $json.split("\n");
        my $data = @lines.map({ "data: $_" }).join("\n");
        "id: $id\n$data\n\n"
    }

    method !store(Str $id, Str $payload) {
        @!history.push({ id => $id, data => $payload });
        if @!history.elems > $!history-size {
            @!history.shift;
        }
    }
}

class StreamableHTTPServerTransport does MCP::Transport::Base::Transport is export {
    has Str $.host = '127.0.0.1';
    has Int $.port = 8080;
    has Str $.path = '/mcp';
    has @.allowed-origins = [];
    has @.protocol-versions = [MCP::Types::LATEST_PROTOCOL_VERSION, '2025-11-25'];
    has Bool $.require-session = False;
    has Bool $.allow-session-delete = True;
    has Int $.stream-history-size = 200;
    has $.oauth-handler;
    has Supplier $!incoming;
    has Supply $!incoming-supply;
    has Bool $!running = False;
    has $!server;
    has %!pending-responses; # id -> { vow => Promise::Vow, init => Bool }
    has %!streams; # id -> Stream
    has @!stream-order;
    has Int $!stream-rr = 0;
    has Str $!session-id;
    has Lock $!send-lock = Lock.new;

    method start(--> Supply) {
        return $!incoming-supply if $!running;

        $!incoming = Supplier.new;
        $!incoming-supply = $!incoming.Supply;
        $!running = True;

        my $application = self!build-router;
        my $server-class = self!cro-class('Cro::HTTP::Server');
        $!server = $server-class.new(
            :$!host,
            :$!port,
            application => $application,
        );
        $!server.start;
        $!incoming-supply;
    }

    method send(MCP::JSONRPC::Message $msg --> Promise) {
        start {
            $!send-lock.protect: {
                given $msg {
                    when MCP::JSONRPC::Response {
                        self!respond-to-pending($msg);
                    }
                    default {
                        self!emit-to-stream($msg);
                    }
                }
            }
        }
    }

    method close(--> Promise) {
        start {
            $!running = False;
            $!server.stop if $!server;
            for %!streams.values -> $stream {
                $stream.supplier.done;
            }
            $!incoming.done if $!incoming;
        }
    }

    method is-connected(--> Bool) {
        $!running
    }

    method !build-router() {
        my &route = self!cro-sub('Cro::HTTP::Router', 'route');
        my &get = self!cro-sub('Cro::HTTP::Router', 'get');
        my &post = self!cro-sub('Cro::HTTP::Router', 'post');
        my &delete = self!cro-sub('Cro::HTTP::Router', 'delete');
        my &content = self!cro-sub('Cro::HTTP::Router', 'content');
        my &request = self!cro-sub('Cro::HTTP::Router', 'request');
        my &response = self!cro-sub('Cro::HTTP::Router', 'response');

        my $self = self;
        my $path = $!path;
        &route({
            my sub path-ok($req --> Bool) {
                my $target = $req.target.split('?', 2)[0] // '';
                $target eq $path
            }

            sub validate-request($req, $resp, &content --> Bool) {
                return False unless path-ok($req);
                return False unless $self!validate-origin($req, $resp, &content);
                return False unless $self!validate-protocol($req, $resp, &content);
                return False unless $self!validate-session($req, $resp, &content);
                return False unless $self!validate-oauth($req, $resp, &content);
                return True;
            }

            if $self.oauth-handler.defined {
                my &get-wk = self!cro-sub('Cro::HTTP::Router', 'get');
                &get-wk(-> '.well-known', 'oauth-protected-resource' {
                    my $resp = &response();
                    $resp.status = 200;
                    &content('application/json', $self.oauth-handler.resource-metadata);
                });
            }

            &get(sub (*@) {
                my $req = &request();
                my $resp = &response();
                unless validate-request($req, $resp, &content) {
                    return;
                }
                unless $self!accepts-sse($req) {
                    $resp.status = 406;
                    return;
                }
                my $last-event-id = $req.header('Last-Event-ID');
                my $stream;
                if $last-event-id.defined && $last-event-id.chars {
                    # Attempt to resume an existing stream
                    $stream = $self!resume-stream($last-event-id);
                    unless $stream.defined {
                        # Stream not found or cannot be resumed
                        $resp.status = 204;
                        return;
                    }
                } else {
                    $stream = $self!open-stream;
                }
                $resp.append-header('Content-Type', 'text/event-stream');
                $resp.append-header('Cache-Control', 'no-cache');
                $resp.append-header('Connection', 'keep-alive');
                $resp.append-header('MCP-Session-Id', $self!session-id) if $self!session-id;
                &content('text/event-stream', $stream.supplier.Supply);
            });

            &post(sub (*@) {
                my $req = &request();
                my $resp = &response();
                unless validate-request($req, $resp, &content) {
                    return;
                }
                unless $self!accepts-post($req) {
                    $resp.status = 406;
                    return;
                }
                unless $self!valid-content-type($req) {
                    $resp.status = 415;
                    &content('application/json', $self!jsonrpc-error("Unsupported Content-Type"));
                    return;
                }

                my $body = await $req.body;
                my Str $json = $self!coerce-body-json($body);
                my $msg;
                {
                    $msg = parse-message($json);
                    CATCH {
                        default {
                            $resp.status = 400;
                            &content('application/json', $self!jsonrpc-error("Invalid JSON-RPC message"));
                            return;
                        }
                    }
                }

                given $msg {
                    when MCP::JSONRPC::Request {
                        my $p = Promise.new;
                        %!pending-responses{$msg.id} = {
                            vow => $p.vow,
                            init => ($msg.method eq 'initialize')
                        };
                        $!incoming.emit($msg);
                        my %payload = await $p;
                        for %payload<headers>.kv -> $k, $v {
                            $resp.append-header($k, $v);
                        }
                        $resp.status = 200;
                        &content('application/json', %payload<body>);
                    }
                    default {
                        $!incoming.emit($msg);
                        $resp.status = 202;
                        return;
                    }
                }
            });

            &delete(sub (*@) {
                my $req = &request();
                my $resp = &response();
                unless validate-request($req, $resp, &content) {
                    return;
                }
                unless $self!allow-session-delete {
                    $resp.status = 405;
                    return;
                }
                if $self!require-session {
                    my $sid = $req.header('MCP-Session-Id');
                    unless $sid && $sid eq $self!session-id {
                        $resp.status = 404;
                        return;
                    }
                }
                $self!terminate-session();
                $resp.status = 204;
                return;
            });
        })
    }

    method !validate-origin($req, $resp, &content --> Bool) {
        my $origin = $req.header('Origin');
        return True unless $origin.defined;
        if @!allowed-origins.elems == 0 {
            $resp.status = 403;
            &content('application/json', self!jsonrpc-error("Invalid Origin"));
            return False;
        }
        if $origin eq any(@!allowed-origins) {
            return True;
        }
        $resp.status = 403;
        &content('application/json', self!jsonrpc-error("Invalid Origin"));
        False
    }

    method !validate-protocol($req, $resp, &content --> Bool) {
        my $ver = $req.header('MCP-Protocol-Version') // DEFAULT_PROTOCOL_FALLBACK;
        if $ver eq any(@!protocol-versions) {
            return True;
        }
        $resp.status = 400;
        &content('application/json', self!jsonrpc-error("Unsupported MCP-Protocol-Version"));
        False
    }

    method !validate-session($req, $resp, &content --> Bool) {
        return True unless $!require-session;
        # When no session exists yet, allow the request (initialization phase)
        return True unless $!session-id.defined;
        # Once a session is established, require matching session ID
        my $sid = $req.header('MCP-Session-Id');
        if $sid && $sid eq $!session-id {
            return True;
        }
        # Per MCP spec: 400 for missing session ID, 404 for unknown session
        if !$sid.defined || $sid eq '' {
            $resp.status = 400;
            &content('application/json', self!jsonrpc-error("Missing MCP-Session-Id header"));
        } else {
            $resp.status = 404;
            &content('application/json', self!jsonrpc-error("Unknown MCP session"));
        }
        False
    }

    method !validate-oauth($req, $resp, &content --> Bool) {
        return True unless $!oauth-handler.defined;
        {
            $!oauth-handler.validate-request($req);
            return True;
            CATCH {
                when X::MCP::OAuth::Unauthorized {
                    $resp.status = 401;
                    $resp.append-header('WWW-Authenticate', $!oauth-handler.www-authenticate-header);
                    &content('application/json', self!jsonrpc-error(.message));
                    return False;
                }
                when X::MCP::OAuth::Forbidden {
                    $resp.status = 403;
                    $resp.append-header('WWW-Authenticate', $!oauth-handler.www-authenticate-scope-header(.scopes));
                    &content('application/json', self!jsonrpc-error(.message));
                    return False;
                }
            }
        }
        False
    }

    method !accepts-post($req --> Bool) {
        my $accept = $req.header('Accept') // '';
        $accept.contains('application/json') && $accept.contains('text/event-stream')
    }

    method !accepts-sse($req --> Bool) {
        my $accept = $req.header('Accept') // '';
        $accept.contains('text/event-stream')
    }

    method !valid-content-type($req --> Bool) {
        my $ct = $req.header('Content-Type') // '';
        $ct.contains('application/json')
    }

    method !coerce-body-json($body --> Str) {
        given $body {
            when Blob|Buf { $body.decode('utf-8') }
            when Str { $body }
            when Hash|Array { to-json($body) }
            default { $body.Str }
        }
    }

    method !respond-to-pending(MCP::JSONRPC::Response $resp) {
        return unless %!pending-responses{$resp.id}:exists;
        my %entry = %!pending-responses{$resp.id}:delete;
        my %headers;
        if %entry<init> {
            $!session-id //= self!new-session-id;
            %headers<MCP-Session-Id> = $!session-id if $!session-id;
        }
        %entry<vow>.keep({
            body => $resp.Hash,
            headers => %headers,
        });
    }

    method !emit-to-stream(MCP::JSONRPC::Message $msg) {
        return unless @!stream-order;
        my $json = $msg.to-json;
        my $stream-id = @!stream-order[$!stream-rr % @!stream-order.elems];
        $!stream-rr++;
        my $stream = %!streams{$stream-id} or return;
        $stream.emit-message($json);
    }

    method !open-stream() {
        my $id = self!new-stream-id;
        my $supplier = Supplier.new;
        my $stream = MCP::Transport::StreamableHTTP::Stream.new(
            id => $id,
            supplier => $supplier,
            history-size => $!stream-history-size
        );
        %!streams{$id} = $stream;
        @!stream-order.push($id);
        $stream.emit-priming;
        $stream
    }

    method !resume-stream(Str $last-event-id) {
        # Event ID format: "streamId:seq"
        my ($stream-id, $seq-str) = $last-event-id.split(':', 2);
        return Nil unless $stream-id.defined && $seq-str.defined;
        my $stream = %!streams{$stream-id};
        return Nil unless $stream.defined;
        my $seq = $seq-str.Int // return Nil;
        # Replay events after the given sequence number
        $stream.replay-from($seq);
        $stream
    }

    method !new-stream-id(--> Str) {
        my $rand = (0..^16).map({ <a b c d e f 0 1 2 3 4 5 6 7 8 9>.pick }).join;
        "s{$rand}"
    }

    method !new-session-id(--> Str) {
        my $rand = (0..^31).map({ <a b c d e f 0 1 2 3 4 5 6 7 8 9>.pick }).join;
        "session-$rand"
    }

    method !session-id(--> Str) { $!session-id }
    method !require-session(--> Bool) { $!require-session }
    method !allow-session-delete(--> Bool) { $!allow-session-delete }

    method !terminate-session() {
        $!session-id = Nil;
    }

    method !jsonrpc-error(Str $message --> Hash) {
        {
            jsonrpc => '2.0',
            error => { code => -32600, message => $message }
        }
    }

    method !cro-class(Str $name) { # UNCOVERABLE
        require ::($name); # UNCOVERABLE
        return ::($name); # UNCOVERABLE
        CATCH { # UNCOVERABLE
            default { # UNCOVERABLE
                die X::MCP::Transport::StreamableHTTP::HTTP.new( # UNCOVERABLE
                    message => "Cro::HTTP is required for StreamableHTTP transport" # UNCOVERABLE
                ); # UNCOVERABLE
            } # UNCOVERABLE
        } # UNCOVERABLE
    } # UNCOVERABLE

    method !cro-sub(Str $module, Str $name) { # UNCOVERABLE
        my $pkg = self!cro-class($module); # UNCOVERABLE
        my $exports = $pkg.WHO<EXPORT>.WHO<DEFAULT>.WHO; # UNCOVERABLE
        # Try regular sub first, then term
        my $sub = $exports{'&' ~ $name} // $exports{'&term:<' ~ $name ~ '>'}; # UNCOVERABLE
        die X::MCP::Transport::StreamableHTTP::HTTP.new( # UNCOVERABLE
            message => "Missing $name in $module" # UNCOVERABLE
        ) unless $sub.defined && $sub ~~ Callable; # UNCOVERABLE
        $sub # UNCOVERABLE
    } # UNCOVERABLE
}

class StreamableHTTPClientTransport does MCP::Transport::Base::Transport is export {
    has Str $.endpoint is required;
    has Str $.protocol-version = MCP::Types::LATEST_PROTOCOL_VERSION;
    has $.client;
    has $.oauth-handler;
    has Supplier $!incoming;
    has Supply $!incoming-supply;
    has Bool $!running = False;
    has Bool $!closing = False;
    has Str $!session-id;
    has Str $!last-event-id;
    has Int $!retry-ms = 1000;
    has Lock $!emit-lock = Lock.new;

    method start(--> Supply) {
        return $!incoming-supply if $!running;
        $!incoming = Supplier.new;
        $!incoming-supply = $!incoming.Supply;
        $!running = True;
        $!client //= self!cro-client;
        start {
            self!sse-loop;
        }
        $!incoming-supply;
    }

    method send(MCP::JSONRPC::Message $msg --> Promise) {
        start {
            my $headers = [
                Accept => DEFAULT_ACCEPT_POST,
                'MCP-Protocol-Version' => $!protocol-version,
            ];
            if $!session-id.defined {
                $headers.push('MCP-Session-Id' => $!session-id);
            }
            if $!oauth-handler.defined {
                my $auth = await $!oauth-handler.authorization-header;
                $headers.push('Authorization' => $auth);
            }
            $!client //= self!cro-client;
            my $resp = await $!client.post(
                $!endpoint,
                headers => $headers,
                content-type => 'application/json',
                body => $msg.to-json
            );

            # Handle 401 by re-authenticating and retrying once
            if $resp.status == 401 && $!oauth-handler.defined {
                await $!oauth-handler.handle-unauthorized;
                my $retry-headers = [
                    Accept => DEFAULT_ACCEPT_POST,
                    'MCP-Protocol-Version' => $!protocol-version,
                ];
                $retry-headers.push('MCP-Session-Id' => $!session-id) if $!session-id.defined;
                my $auth = await $!oauth-handler.authorization-header;
                $retry-headers.push('Authorization' => $auth);
                $resp = await $!client.post(
                    $!endpoint,
                    headers => $retry-headers,
                    content-type => 'application/json',
                    body => $msg.to-json
                );
            }

            self!capture-session-id($resp);
            return if $resp.status == 202;

            # Per MCP spec: 404 means session expired, must reinitialize
            if $resp.status == 404 && $!session-id.defined {
                $!session-id = Nil;
                die X::MCP::Transport::StreamableHTTP::Protocol.new(
                    message => "Session expired, reinitialization required"
                );
            }

            my $ctype = $resp.header('Content-Type') // '';
            if $ctype.contains('text/event-stream') {
                await self!consume-sse($resp.body-byte-stream);
            } else {
                my $body = await $resp.body;
                if $body.defined {
                    my $json =
                        $body ~~ Str ?? $body
                        !! $body ~~ Hash|Array ?? to-json($body)
                        !! $body ~~ Blob|Buf ?? $body.decode('utf-8')
                        !! $body.Str;
                    self!emit-json($json);
                }
            }
        }
    }

    method close(--> Promise) {
        start {
            $!closing = True;
            $!running = False;
            $!incoming.done if $!incoming;
        }
    }

    #| Terminate the session by sending DELETE request
    method terminate-session(--> Promise) {
        start {
            return False unless $!session-id.defined;
            my @headers = (
                'MCP-Protocol-Version' => $!protocol-version,
                'MCP-Session-Id' => $!session-id,
            );
            if $!oauth-handler.defined {
                my $auth = await $!oauth-handler.authorization-header;
                @headers.push('Authorization' => $auth);
            }
            # Use a fresh client to avoid conflicts with SSE connection
            my $delete-client = self!cro-client;
            my $result = False;
            {
                my $resp = await $delete-client.delete($!endpoint, headers => @headers);
                if $resp.status == 204 || $resp.status == 200 {
                    $!session-id = Nil;
                    $result = True;
                }
                CATCH { default { } }
            }
            $result
        }
    }

    method is-connected(--> Bool) {
        $!running
    }

    method !sse-loop() {
        loop {
            last if $!closing;
            my @headers = (
                Accept => DEFAULT_ACCEPT_SSE,
                'MCP-Protocol-Version' => $!protocol-version,
            );
            if $!session-id.defined {
                @headers.push('MCP-Session-Id' => $!session-id);
            }
            if $!last-event-id.defined {
                @headers.push('Last-Event-ID' => $!last-event-id);
            }
            if $!oauth-handler.defined {
                my $auth = await $!oauth-handler.authorization-header;
                @headers.push('Authorization' => $auth);
            }
            $!client //= self!cro-client;
            my $resp = await $!client.get($!endpoint, headers => @headers);
            self!capture-session-id($resp);
            if $resp.status == 405 {
                last;
            }
            if $resp.status >= 400 {
                last;
            }
            await self!consume-sse($resp.body-byte-stream);
            CATCH { default { } }
            last if $!closing;
            sleep $!retry-ms / 1000;
        }
    }

    method !consume-sse(Supply $bytes --> Promise) {
        start {
            my $buffer = '';
            my $event-id;
            my @data;
            my $retry;

            react {
                whenever $bytes -> $chunk {
                    $buffer ~= $chunk.decode('utf-8');
                    loop {
                        my $idx = $buffer.index("\n");
                        last unless $idx.defined;
                        my $line = $buffer.substr(0, $idx);
                        $buffer = $buffer.substr($idx + 1);
                        $line = $line.subst(/\r$/, '');
                        if $line eq '' {
                            if @data {
                                my $data = @data.join("\n");
                                self!emit-json($data);
                            }
                            $event-id.defined && ($!last-event-id = $event-id);
                            @data = ();
                            $event-id = Nil;
                            $retry = Nil;
                            next;
                        }
                        next if $line.substr(0,1) eq ':';
                        my ($field, $value) = $line.split(':', 2);
                        $value = $value.subst(/^ /, '') if $value.defined;
                        given $field {
                            when 'id' { $event-id = $value }
                            when 'data' { @data.push($value // '') }
                            when 'retry' {
                                $retry = $value.Int;
                                $!retry-ms = $retry if $retry > 0;
                            }
                            default { }
                        }
                    }
                }
            }
        }
    }

    method !emit-json(Str $json) {
        return unless $json.defined && $json.chars;
        my $msg;
        {
            $msg = parse-message($json);
            CATCH { default { return } }
        }
        $!emit-lock.protect: { $!incoming.emit($msg) if $msg.defined }
    }

    method !capture-session-id($resp) {
        my $sid = $resp.header('MCP-Session-Id');
        $!session-id = $sid if $sid.defined && $sid.chars;
    }

    method !cro-client() { # UNCOVERABLE
        my $client-class = self!cro-class('Cro::HTTP::Client'); # UNCOVERABLE
        $client-class.new # UNCOVERABLE
    } # UNCOVERABLE

    method !cro-class(Str $name) { # UNCOVERABLE
        require ::($name); # UNCOVERABLE
        return ::($name); # UNCOVERABLE
        CATCH { # UNCOVERABLE
            default { # UNCOVERABLE
                die X::MCP::Transport::StreamableHTTP::HTTP.new( # UNCOVERABLE
                    message => "Cro::HTTP is required for StreamableHTTP transport" # UNCOVERABLE
                ); # UNCOVERABLE
            } # UNCOVERABLE
        } # UNCOVERABLE
    } # UNCOVERABLE
}
