unit module Math::SalvoCombatModeling;

use JSON::Fast;

my constant @MODELKEYS = <units offense-matrix defense-matrix>;

sub force-name-str($x) {
    $x ~~ Str ?? $x !! $x.gist
}

sub validate-force-spec(@spec, Str:D $name) {
    die "$name must be a 2-element positional value: [force-name, unit-count]"
      unless @spec.elems == 2;
    die "$name unit-count must be a positive integer"
      unless @spec[1] ~~ Int:D and @spec[1] > 0;
    die "$name force-name is not valid"
      unless salvo-force-name-q(@spec[0]);
}

sub unit-symbol($force, Int:D $i) {
    "{force-name-str($force)}[$i]"
}

my %to-greek = <beta gamma delta capitalpsi rho sigma tau capitaltheta zeta> Z=> <ß γ δ Ψ ρ σ τ Θ 𝛇>;

sub var-call(Str:D $name, *@args, *%args) {
    my $greek = %args<latex> // %args<greek> // False;
    ($greek ?? %to-greek{$name} !! $name) ~ '(' ~ @args.map(*.gist).join(',') ~ ')'
}

sub param-value(&param, Str:D $name, *@args) {
    &param($name, |@args)
}

sub unit-value($force, Int:D $i, :&param, :&units) {
    return units($force, $i) if &units.defined;
    return param-value(&param, 'units', $force, $i) if &param.defined;
    unit-symbol($force, $i)
}

sub safe-div($numerator, $denominator, Bool:D :$latex = False) {
    return "\\frac\{$numerator\}\{$denominator\}" if $latex;
    return ($numerator ~~ Numeric && $denominator ~~ Numeric)
      ?? $numerator / $denominator
      !! "($numerator)/($denominator)"
}

sub safe-sub($lhs, $rhs, Bool:D :$latex = False) {
    return "($lhs) - ($rhs)" if $latex;
    return ($lhs ~~ Numeric && $rhs ~~ Numeric)
      ?? $lhs - $rhs
      !! "($lhs)-($rhs)"
}

sub safe-mul(*@factors, Bool:D :$latex = False) {
    return do if @factors.all ~~ Numeric {
        [*] @factors
    } else {
        if $latex {
            @factors.join(' \times ')
        } else {
            @factors.map({ $ ~~ Numeric ?? $_.gist !! "($_)" }).join('*')
        }
    }
}

sub sum-terms(*@terms) {
    if @terms.all ~~ Numeric {
        [+] @terms;
    } else {
        @terms.map({ $ ~~ Numeric ?? $_.gist !! "($_)" }).join(' + ');
    }
}

multi sub salvo-force-name-q($x --> Bool:D) is export {
    !($x ~~ Positional) && !($x ~~ Associative) && !($x ~~ Callable)
}

multi sub salvo-force-name-q(Any:U --> Bool:D) is export { False }

sub salvo-model-q(%model --> Bool:D) is export {
    return False unless %model.elems > 0;
    for %model.values -> $side {
        return False unless $side ~~ Associative;
        return False unless $side.keys.sort eqv @MODELKEYS.sort;
    }
    True
}

proto sub salvo-variable(|) is export {*}

multi sub salvo-variable(Str:D $symbol, $x, $y, Int:D $p, Int:D $q --> Pair:D) {
    given $symbol.lc {
        when $_ ∈ <beta β> {
            var-call('beta', $x, $y, $p, $q) =>
              "beta: Offensive combat potential of {$x.gist}[$p] units against {$y.gist}[$q]. (hits /shooting unit)"
        }
        when $_ ∈ <capitalpsi psi Ψ> {
            var-call('capitalpsi', $x, $y, $p, $q) =>
              "capitalpsi: Fraction of {$x.gist}[$p] units that engage {$y.gist}[$q] units. [0,1]"
        }
        when $_ ∈ <gamma γ> {
            var-call('gamma', $x, $y, $p, $q) =>
              "gamma: Defensive combat power of side {$x.gist}[$p] against {$y.gist}[$q] units, (shots / defending units)"
        }
        when $_ ∈ <capitaltheta theta ϴ> {
            var-call('capitaltheta', $x, $y, $p, $q) =>
              "capitaltheta: Fraction of {$x.gist}[$p] units that engage {$y.gist}[$q] units. [0,1]"
        }
        when $_ ∈ <sigma σ> {
            var-call('sigma', $x, $y, $p, $q) =>
              "sigma: Scouting effectiveness of unit {$x.gist}[$p] against {$y.gist}[$q]. [0,1]"
        }
        when $_ ∈ <tau τ> {
            var-call('tau', $x, $y, $p, $q) =>
              "tau: Training effectiveness of unit {$x.gist}[$p] against {$y.gist}[$q]. [0,1]"
        }
        when $_ ∈ <rho ⍴> {
            var-call('rho', $x, $y, $p, $q) =>
              "rho: Distraction factor of unit {$x.gist}[$p] against {$y.gist}[$q]. [0,1]"
        }
        when $_ ∈ <curlyepsilon epsilon ε> {
            var-call('curlyepsilon', $x, $y, $p, $q) =>
              "curlyepsilon: Offensive effectiveness of {$x.gist}[$p] against {$y.gist}[$q]. [0,1]"
        }
        when $_ ∈ <delta δ> {
            var-call('delta', $x, $y, $p, $q) =>
              "delta: Defender alertness or readiness of unit {$x.gist}[$p] against {$y.gist}[$q]. [0,1]"
        }
        default {
            die "Unsupported 4-index variable symbol '$symbol'"
        }
    }
}

multi sub salvo-variable($B, Int:D $j --> Pair:D) {
    unit-symbol($B, $j) =>
      "Number of {force-name-str($B)} units of type $j."
}

multi sub salvo-variable(Str:D $symbol, $A, Int:D $i --> Pair:D) {
    given $symbol.lc {
        when 'zeta' {
            var-call('zeta', $A, $i) =>
              "zeta: Staying power of {$A.gist}[$i] unit, (hits)"
        }
        default {
            die "Unsupported 2-index variable symbol '$symbol'"
        }
    }
}

multi sub salvo-variable(Str:D $symbol, $A --> Pair:D) {
    given $symbol.lc {
        when 'capitaldelta' {
            var-call('capitaldelta', $A) =>
              "The number of {$A.gist} units put out of action"
        }
        default {
            die "Unsupported 1-index variable symbol '$symbol'"
        }
    }
}

sub salvo-variable-rules(@force-a, @force-b --> Array:D) is export {
    validate-force-spec(@force-a, 'first force');
    validate-force-spec(@force-b, 'second force');

    my ($A, $m) = @force-a;
    my ($B, $n) = @force-b;
    my @vars-a = <beta capitalpsi gamma capitaltheta zeta sigma tau rho curlyepsilon delta>;
    my @rules;

    for @vars-a -> $v {
        for 1 .. $m -> $i {
            for 1 .. $n -> $j {
                @rules.push: $v eq 'zeta'
                  ?? salvo-variable($v, $A, $i)
                  !! salvo-variable($v, $A, $B, $i, $j);
            }
        }
    }
    for @vars-a -> $v {
        for 1 .. $n -> $i {
            for 1 .. $m -> $j {
                @rules.push: $v eq 'zeta'
                  ?? salvo-variable($v, $B, $i)
                  !! salvo-variable($v, $B, $A, $i, $j);
            }
        }
    }
    my %seen;
    @rules.grep({
        my $k = .key;
        if %seen{$k}:exists {
            False
        } else {
            %seen{$k} = True;
            True
        }
    }).Array
}

sub salvo-terms(@force-a, @force-b, :&param, :&units --> Array:D) is export {
    validate-force-spec(@force-a, 'first force');
    validate-force-spec(@force-b, 'second force');
    my ($A, $m) = @force-a;
    my ($B, $n) = @force-b;

    my @matrix;
    for 1 .. $m -> $i {
        my @row;
        for 1 .. $n -> $j {
            my $lhs = &param.defined
              ?? safe-mul(
                    param-value(&param, 'sigma', $B, $A, $j, $i),
                    param-value(&param, 'tau', $B, $A, $j, $i),
                    param-value(&param, 'rho', $A, $B, $i, $j),
                    param-value(&param, 'beta', $B, $A, $j, $i),
                    param-value(&param, 'capitalpsi', $B, $A, $j, $i),
                    unit-value($B, $j, :&param, :&units)
                )
              !! safe-mul(
                    var-call('sigma', $B, $A, $j, $i),
                    var-call('tau', $B, $A, $j, $i),
                    var-call('rho', $A, $B, $i, $j),
                    var-call('beta', $B, $A, $j, $i),
                    var-call('capitalpsi', $B, $A, $j, $i),
                    unit-value($B, $j, :&param, :&units)
                );

            my $rhs = &param.defined
              ?? safe-mul(
                    param-value(&param, 'delta', $A, $B, $i, $j),
                    param-value(&param, 'tau', $A, $B, $i, $j),
                    param-value(&param, 'gamma', $A, $B, $i, $j),
                    param-value(&param, 'capitaltheta', $A, $B, $i, $j),
                    unit-value($A, $i, :&param, :&units)
                )
              !! safe-mul(
                    var-call('delta', $A, $B, $i, $j),
                    var-call('tau', $A, $B, $i, $j),
                    var-call('gamma', $A, $B, $i, $j),
                    var-call('capitaltheta', $A, $B, $i, $j),
                    unit-value($A, $i, :&param, :&units)
                );

            my $zeta = &param.defined
              ?? param-value(&param, 'zeta', $A, $i)
              !! var-call('zeta', $A, $i);

            @row.push: safe-div(safe-sub($lhs, $rhs), $zeta);
        }
        @matrix.push: @row.Array;
    }
    @matrix.Array
}

sub salvo-damage(@force-a, @force-b, :&param, :&units) is export {
    my @terms = salvo-terms(@force-a, @force-b, :&param, :&units).flat;
    sum-terms(@terms)
}

sub heterogeneous-salvo-model(
    @force-a,
    @force-b,
    Bool:D :$offensive-effectiveness-terms = False,
    :&param,
    :&units,
    Bool:D :$latex = False,
    --> Hash:D
) is export {
    validate-force-spec(@force-a, 'first force');
    validate-force-spec(@force-b, 'second force');
    my ($A, $m) = @force-a;
    my ($B, $n) = @force-b;

    my $name-a = force-name-str($A);
    my $name-b = force-name-str($B);
    my @vec-a = (1 .. $m).map({ unit-value($A, $_, :&param, :&units) }).Array;
    my @vec-b = (1 .. $n).map({ unit-value($B, $_, :&param, :&units) }).Array;

    my @mat-offense-a;
    for 1 .. $m -> $i {
        my @row;
        for 1 .. $n -> $j {
            if &param.defined {
                my $off = $offensive-effectiveness-terms
                  ?? param-value(&param, 'curlyepsilon', $B, $A, $j, $i)
                  !! safe-mul(
                        param-value(&param, 'sigma', $B, $A, $j, $i),
                        param-value(&param, 'tau', $B, $A, $j, $i),
                        param-value(&param, 'rho', $A, $B, $i, $j)
                    );
                @row.push: safe-div(
                  safe-mul(
                    param-value(&param, 'beta', $B, $A, $j, $i),
                    $off,
                    param-value(&param, 'capitalpsi', $B, $A, $j, $i),
                  ),
                  param-value(&param, 'zeta', $A, $i)
                );
            } else {
                my $off = $offensive-effectiveness-terms
                  ?? var-call('curlyepsilon', $B, $A, $j, $i, :$latex)
                  !! safe-mul(
                        var-call('sigma', $B, $A, $j, $i, :$latex),
                        var-call('tau', $B, $A, $j, $i, :$latex),
                        var-call('rho', $A, $B, $i, $j, :$latex),
                        :$latex
                    );
                @row.push: safe-div(
                  safe-mul(
                    var-call('beta', $B, $A, $j, $i, :$latex),
                    $off,
                    var-call('capitalpsi', $B, $A, $j, $i, :$latex),
                    :$latex
                  ),
                  var-call('zeta', $A, $i, :$latex),
                  :$latex
                );
            }
        }
        @mat-offense-a.push: @row.Array;
    }

    my @diag-a;
    for 1 .. $m -> $i {
        my @sum-parts;
        for 1 .. $n -> $j {
            if &param.defined {
                @sum-parts.push: safe-div(
                  safe-mul(
                    param-value(&param, 'gamma', $A, $B, $i, $j),
                    param-value(&param, 'delta', $A, $B, $i, $j),
                    param-value(&param, 'capitaltheta', $A, $B, $i, $j),
                    param-value(&param, 'tau', $A, $B, $i, $j)
                  ),
                  param-value(&param, 'zeta', $A, $i)
                );
            } else {
                @sum-parts.push: safe-div(
                  safe-mul(
                    var-call('gamma', $A, $B, $i, $j, :$latex),
                    var-call('delta', $A, $B, $i, $j, :$latex),
                    var-call('capitaltheta', $A, $B, $i, $j, :$latex),
                    var-call('tau', $A, $B, $i, $j, :$latex),
                    :$latex
                  ),
                  var-call('zeta', $A, $i, :$latex),
                  :$latex
                );
            }
        }
        @diag-a.push: sum-terms(@sum-parts);
    }

    my @mat-defense-a;
    for 1 .. $m -> $i {
        my @row = (^$m).map({ 0 }).Array;
        @row[$i - 1] = @diag-a[$i - 1];
        @mat-defense-a.push: @row.Array;
    }

    my @mat-offense-b;
    for 1 .. $n -> $i {
        my @row;
        for 1 .. $m -> $j {
            if &param.defined {
                my $off = $offensive-effectiveness-terms
                  ?? param-value(&param, 'curlyepsilon', $A, $B, $j, $i)
                  !! safe-mul(
                        param-value(&param, 'sigma', $A, $B, $j, $i),
                        param-value(&param, 'tau', $A, $B, $j, $i),
                        param-value(&param, 'rho', $B, $A, $i, $j)
                    );
                @row.push: safe-div(
                  safe-mul(
                    param-value(&param, 'beta', $A, $B, $j, $i),
                    $off,
                    param-value(&param, 'capitalpsi', $A, $B, $j, $i)
                  ),
                  param-value(&param, 'zeta', $B, $i)
                );
            } else {
                my $off = $offensive-effectiveness-terms
                  ?? var-call('curlyepsilon', $A, $B, $j, $i, :$latex)
                  !! safe-mul(
                        var-call('sigma', $A, $B, $j, $i, :$latex),
                        var-call('tau', $A, $B, $j, $i, :$latex),
                        var-call('rho', $B, $A, $i, $j, :$latex),
                        :$latex
                    );
                @row.push: safe-div(
                  safe-mul(
                    var-call('beta', $A, $B, $j, $i, :$latex),
                    $off,
                    var-call('capitalpsi', $A, $B, $j, $i, :$latex),
                    :$latex
                  ),
                  var-call('zeta', $B, $i, :$latex),
                  :$latex
                );
            }
        }
        @mat-offense-b.push: @row.Array;
    }

    my @diag-b;
    for 1 .. $n -> $i {
        my @sum-parts;
        for 1 .. $m -> $j {
            if &param.defined {
                @sum-parts.push: safe-div(
                  safe-mul(
                    param-value(&param, 'gamma', $B, $A, $i, $j),
                    param-value(&param, 'delta', $B, $A, $i, $j),
                    param-value(&param, 'capitaltheta', $B, $A, $i, $j),
                    param-value(&param, 'tau', $B, $A, $i, $j)
                  ),
                  param-value(&param, 'zeta', $B, $i)
                );
            } else {
                @sum-parts.push: safe-div(
                  safe-mul(
                    var-call('gamma', $B, $A, $i, $j, :$latex),
                    var-call('delta', $B, $A, $i, $j, :$latex),
                    var-call('capitaltheta', $B, $A, $i, $j, :$latex),
                    var-call('tau', $B, $A, $i, $j, :$latex)
                  ),
                  var-call('zeta', $B, $i, :$latex)
                );
            }
        }
        @diag-b.push: sum-terms(@sum-parts);
    }

    my @mat-defense-b;
    for 1 .. $n -> $i {
        my @row = (^$n).map({ 0 }).Array;
        @row[$i - 1] = @diag-b[$i - 1];
        @mat-defense-b.push: @row.Array;
    }

    {
        $name-a => {
            units => @vec-a.Array,
            offense-matrix => @mat-offense-a.Array,
            defense-matrix => @mat-defense-a.Array
        },
        $name-b => {
            units => @vec-b.Array,
            offense-matrix => @mat-offense-b.Array,
            defense-matrix => @mat-defense-b.Array
        }
    }
}

sub salvo-model(|c) is export {
    heterogeneous-salvo-model(|c)
}

#==========================================================
# Definitions reading
#==========================================================

our sub resources {
    %?RESOURCES
}

my constant @LANGUAGES = <bulgarian english russian>;

sub salvo-notion-definitions(Str:D $lang = 'English' --> Mu) is export {
    die "Unsupported language '$lang'. Expected one of: {@LANGUAGES.join(', ')}"
    unless $lang.lc ∈ @LANGUAGES;

    my $file = "Definitions{$lang.tc}.json";
    return from-json(slurp(%?RESOURCES{$file}.IO));
}
