#!/usr/bin/env raku
use v6.d;

use lib <. lib>;
use LLM::Functions;
use LLM::Prompts;
use Text::SubParsers;
use LLM::Tooling;
use LLM::Functions::Tooled;
use LLM::Functions::TooledGemini;
use LLM::Functions::TooledChatGPT;
use Chemistry::Stoichiometry;

use JSON::Fast;

# ==============================================================

# Assuming you already have the tool implementations:
#   sub molecular-mass(%args --> ...) { ... }
#   sub balance-chemical-equation(%args --> ...) { ... }

my @tool-objects =
        LLM::Tool.new(&molecular-mass),
        LLM::Tool.new(&balance-chemical-equation)
        ;

# Optional: If your LLM::Tool objects don’t expose .Hash('Gemini'),
# pass @toolspecs explicitly (same structure as your notebook’s @tools):
my @tool-specs = (
{
    :name("molecular-mass"),
    :description("Convert chemical compound formula into molecule mass."),
    :parameters({
        :type("object"),
        :properties( {"\$spec" => { :description("A molecule formula or list"), :type("string") }} ),
        :required(["\$spec"]),
    }),
},
{
    :name("balance-chemical-equation"),
    :description("Balance the given chemical equation."),
    :parameters({
        :type("object"),
        :properties( {"\$spec" => { :description("A chemical equation"), :type("string") }} ),
        :required(["\$spec"]),
    }),
},
);

# A single prompt or an array of prompts:
my $input = "What are the masses of SO2, O3, and C2H5OH? Also balance: C2H5OH + O2 = H2O + CO2";
#my $input = "How many molecules a kilogram of water has?";

# Run with Gemini:
#`[
my $conf-gemini-flash = llm-configuration('Gemini', model => 'gemini-2.0-flash', :8192max-tokens, temperature => 0.4);

my $obj = LLM::Functions::TooledGemini.new(conf => $conf-gemini-flash);

# :@tool-specs
my $final = $obj.eval(
        $input,
        :@tool-objects,
        format => 'text');
]

# Run with ChatGPT:
#`[
my $conf41-mini = llm-configuration('ChatGPT', model => 'gpt-4.1-mini', :8192max-tokens, temperature => 0.4);

my $obj = LLM::Functions::TooledChatGPT.new(conf => $conf41-mini);

# :@tool-specs
my $final = $obj.eval(
        $input,
        :@tool-objects,
        format => 'text'):echo;
]

my $conf41-mini = llm-configuration('ChatGPT', model => 'gpt-4.1-mini', :8192max-tokens, temperature => 0.4);
my $conf-gemini-flash = llm-configuration('Gemini', model => 'gemini-2.0-flash', :8192max-tokens, temperature => 0.4);

my $final = llm-synthesize-with-tools(
        [$input, llm-prompt('NothingElse')('JSON')],
        @tool-objects,
        e => $conf41-mini,
        form => sub-parser('JSON'):drop):echo;

# Get the assistant text(s)
say $final;
