#!/usr/bin/env raku
use v6.d;

use LLM::Tooling;
use LLM::Functions;
use WWW::Gemini;
use JSON::Fast;

# Helper: extract ToolRequests from a Gemini candidate content
sub extract-tool-requests(%assistant-content) {
    my @requestObjects;
    if %assistant-content<parts> {
        for |%assistant-content<parts> -> %part {
            if %part<functionCall> && %part<functionCall><name> {
                my $name = %part<functionCall><name>;
                my %args = (%part<functionCall><args> // {}).Hash;
                @requestObjects.push: LLM::ToolRequest.new($name, %args);
            }
        }
    }
    return @requestObjects;
}


#| Synthesize with tools in a loop until the LLM returns a final, non-tool response.
#| Arg 1: Str or Array[Str] with the user messages (first message(s)).
#| Arg 2: Array of LLM::Tool objects (callable tool implementations).
#| Named:
#|   :$model = "gemini-2.0-flash"
#|   :@toolspecs  # Gemini tool specs (if omitted, try to derive from @tool-objs)
#|   :%tool-config = { functionCallingConfig => { mode => "ANY" } }
#|   :$max-loops = 8  # safety cap
#|   :$format = Nil   # e.g. "raku" if you want formatted code back
sub llm-synthesize-with-tools(
        Cool  $first-arg,                      # Str or Array[Str]
        @tool-objects where { .all ~~ LLM::Tool:D },# list of LLM::Tool objects
        :$model       = "gemini-2.0-flash",
        :@toolspecs   = [],                           # optional Gemini tool specs
        :%tool-config = { functionCallingConfig => { mode => "ANY" } },
        :$max-loops   = 8,
        :$format      = Whatever,
                               ) is export {

    # 1) Normalize initial user messages -> Gemini "messages"
    my $prompt = $first-arg ~~ (Array:D | List:D | Seq:D) ?? $first-arg.join("\n") !! $first-arg.Str;
    my @messages = [%( role => 'user', parts => [ %( text => $prompt ), ] ), ];

    # 2) Get tool specs for Gemini (either provided or derived from tool objs)
    my @tool-specs = @toolspecs.elems
            ?? @toolspecs
            !! @tool-objects.map({
                # Many LLM::Tool implementations support this; if not, pass @toolspecs yourself.
                $_.Hash('Gemini')
            });

    # First call
    my $response = gemini-generate-content(
            @messages,
            :$model,
            tools => @tool-specs,
            :%tool-config,
            format => 'hash');

    note ('loops' => 0);
    note (:$response);

    # Safety loop
    my $loops = 0;

    loop {
        $loops++;
        note (:$loops);

        die "llm-synthesize-with-tools: exceeded max loops ($max-loops)."
        if $loops > $max-loops;

        note (:$response);

        # Extract first candidate’s content
        my %assistant-message = $response<candidates>[0]<content>;

        # 4) If the LLM returned tool-call(s), run them locally and continue
        my @requests = extract-tool-requests(%assistant-message);

        if @requests.elems {
            # 4.1–4.3 Compute with the tools and add functionResponse messages
            my @funcParts = @requests.map({ generate-llm-tool-response(@tool-objects, $_) })».Hash('Gemini');

            # Make and add the user response
            my %function-response =
                    role => 'user',
                    parts => @funcParts;

            @messages.push(%function-response);

            # 4.4 Extend conversation:
            #  - include the assistant message that requested tools
            #  - include our functionResponse messages with results
            @messages.push(%assistant-message);
            @messages.push(%function-response);

            # 4.5 goto loop (send back to the LLM)
            # Send the second request with function result
            $response = gemini-generate-content(
                    @messages,
                    tools => @tool-specs,
                    :$model,
                    format => "hash");

            next;

        }

        # 5) No tool calls — return the last LLM result (entire candidate content)

        return do if $format ~~ Str:D && $format.lc ∈ <text values> {
            $response<candidates>[0]<content><parts>».<text>.join("\n");
        } else {
            $response<candidates>
        }
    }
}

# ==============================================================

use Chemistry::Stoichiometry;

my $model = "gemini-2.0-flash";
my $format = 'json';

# Assuming you already have the tool implementations:
#   sub molecular-mass(%args --> ...) { ... }
#   sub balance-chemical-equation(%args --> ...) { ... }

my @tool-objects =
        LLM::Tool.new(&molecular-mass),
        LLM::Tool.new(&balance-chemical-equation)
        ;

# Optional: If your LLM::Tool objects don’t expose .Hash('Gemini'),
# pass @toolspecs explicitly (same structure as your notebook’s @tools):
my @toolspecs = (
{
    :name("molecular-mass"),
    :description("Convert chemical compound formula into molecule mass."),
    :parameters({
        :type("object"),
        :properties( {"\$spec" => { :description("A molecule formula or list"), :type("string") }} ),
        :required(["\$spec"]),
    }),
},
{
    :name("balance-chemical-equation"),
    :description("Balance the given chemical equation."),
    :parameters({
        :type("object"),
        :properties( {"\$spec" => { :description("A chemical equation"), :type("string") }} ),
        :required(["\$spec"]),
    }),
},
);

# A single prompt or an array of prompts:
#my $input = "What are the masses of SO2, O3, and C2H5OH? Also balance: C2H5OH + O2 = H2O + CO2";
my $input = "How many molecules a kilogram of water has?";

# Run:
my $final = llm-synthesize-with-tools(
        $input,
        @tool-objects,
        :$model,
        :@toolspecs,
        format => 'text');

# Get the assistant text(s)
say $final;
