#!/usr/bin/env raku
use v6.d;

# use lib <. lib>;

use LLM::Functions;
use LLM::Tooling;
use JSON::Fast;

use Data::Reshapers;
use Data::TypeSystem;

#===========================================================

#| Powers the spinning disco ball.
sub power-disco-ball-impl(
        Int:D $power #= Whether to turn the disco ball on or off.
                          ) returns Hash {
    return { status => "Disco ball powered " ~ ($power ?? 'on' !! 'off') };
}
#= A status dictionary indicating the current state.

#| Play some music matching the specified parameters.
sub start-music-impl(
        Int:D $energetic, #=  Whether the music is energetic or not.
        Int:D $loud       #= Whether the music is loud or not.
                     ) returns Hash {
    my $music-type = $energetic ?? 'energetic' !! 'chill';
    my $volume = $loud ?? 'loud' !! 'quiet';
    return { music_type => $music-type, volume => $volume };
    #= A dictionary containing the music settings.
}

#| Dim the lights.
sub dim-lights-impl(
        Numeric:D $brightness #= The brightness of the lights, 0.0 is off, 1.0 is full.
                    ) returns Hash {
    return { brightness => $brightness };
}
#= A dictionary containing the new brightness setting.

#===========================================================

say llm-tool-definition(&dim-lights-impl, format => 'json');

#===========================================================

my @tools =
        {
            :name("power-disco-ball-impl"),
            :description("Powers the spinning disco ball."),
            :parameters(
            {
                :type("object")
                :properties( {"\$power" => {:description("Whether to turn the disco ball on or off."), :type("integer")}}),
                :required(["\$power"]),
            }),
        },
        {
            :name("start-music-impl"),
            :description("Play some music matching the specified parameters."),
            :parameters(
            {
                :type("object")
                :properties({
                    "\$energetic" => {:description("Whether the music is energetic or not."), :type("integer")},
                    "\$loud" => {:description("Whether the music is loud or not."), :type("integer")}
                }),
                :required(["\$energetic", "\$loud"]),
            }),
        },
        {
            :name("dim-lights-impl"),
            :description("Dim the lights."),
            :parameters(
            {
                :type("object")
                :properties({"\$brightness" => {:description("The brightness of the lights, 0.0 is off, 1.0 is full."), :type("number")}}),
                :required(["\$brightness"]),
            }),
        };

say deduce-type(@tools);

#===========================================================

my %toolConfig =
        functionCallingConfig => {
            mode => "ANY",
            allowedFunctionNames => <power-disco-ball-impl start-music-impl dim-lights-impl>
        };

#===========================================================

my $model = "gemini-2.0-flash";
my $conf = llm-configuration('gemini', :$model, :4096max-tokens, temperature => 0.45, :@tools);

say $conf.Hash.elems;

$conf.tools = @tools;

my @toolObjects = [&power-disco-ball-impl, &start-music-impl, &dim-lights-impl].map({ LLM::Tool.new($_) });

.say for @toolObjects;

#===========================================================

# Make a chat evaluator object
my $llmEvalObj = LLM::Functions::EvaluatorChatGemini.new(:$conf);

# Make a new chat object that uses the chat evaluator object
my $chat = LLM::Functions::Chat.new(llm-evaluator => $llmEvalObj, chat-id => 'new-parallel-chat-' ~ now);

# User prompt
my $prompt = 'Turn this place into a party!';

# First call
my @choices = $chat.eval($prompt, format => 'choices'):echo;

say '@choices' => @choices.raku;

my @llmToolRequests = @choices.grep(*<finish_reason> eq 'tool_calls');

say ('@llmToolRequests' => @llmToolRequests.raku);

say @llmToolRequests.head<message><tool_calls>;

say '=' x 100;

#`[
my $k = 0;
while @llmToolRequests {

    say '-' x 100;
    say $k++;
    say '-' x 100;

    #$chat.messages.pop;

    $chat.messages.push: {
        role => 'assistant',
        tool_calls => @llmToolRequests.head<message><tool_calls>,
        content => ''
    }

    #note ('@llmToolRequests.head' => @llmToolRequests.head».gist);

    my @toolRequests = llm-tool-requests(@llmToolRequests.head);

    #note (@toolRequests.head);

    my $genRes = generate-llm-tool-response(@toolObjects, @toolRequests.head);

    my %message =
            role => 'tool',
            content => $genRes.head.output,
            tool_call_id => @llmToolRequests.head<message><tool_calls>.head<id>;

    @choices = $chat.eval(%message, format => 'choices');
    @llmToolRequests = @choices.grep(*<finish_reason> eq 'tool_calls');

    say '@llmToolRequests.elems : ', @llmToolRequests.elems;
    #say 'Last response : ', $chat.messages.tail».gist;
}
]