# Implementation::Loader Enhancements for Walker Discovery

**Date**: 2025-01-27  
**Context**: Analysis of using `Implementation::Loader` for `Qwiratry::WalkerFactory` discovery mechanism  
**Source**: `Implementation::Loader` at https://raw.githubusercontent.com/wayland/raku-Implementation-Loader/refs/heads/main/lib/Implementation/Loader.rakumod

## Proposed Enhancements

### 1. Enhanced `load-library` Method

**Current signature:**
```raku
method load-library(Str :$type = 'Database::Storage::Memory', *%parameters)
```

**Proposed enhanced signature:**
```raku
method load-library(
    Str :$module-name,
    Str :$type,
    Str :$does,
    Bool :$return-type = False,  # Return type object instead of instance
    *%parameters
)
```

**Benefits:**
- **Separation of concerns**: Module name can differ from type name
- **Type safety**: Verify role composition before instantiation
- **Discovery support**: Return type objects when instances aren't needed
- **Better error messages**: Clear failure when role check fails

**Implementation:**
```raku
method load-library(
    Str :$module-name,
    Str :$type,
    Str :$does,
    Bool :$return-type = False,
    *%parameters
) {
    # Determine which module to load
    my $module-to-load = $module-name // $type;
    
    # Ensure we have a type name
    my $type-name = $type // $module-to-load;
    
    %!library-locks{$module-to-load}:exists or %!library-locks{$module-to-load} = Lock.new();
    
    my $result = %!library-locks{$module-to-load}.protect: {
        # Load the module
        my \M = (require ::($module-to-load));
        
        # If type name differs from module name, resolve it
        my \Type = $type-name eq $module-to-load ?? M !! ::($type-name);
        
        # Verify role composition if specified
        if $does.defined {
            my \Role = ::($does);
            unless Type.^does(Role) {
                die X::TypeCheck::Role.new(
                    operation => "load-library",
                    got => Type,
                    expected => Role,
                    message => "Type {Type.^name} does not do role {$does}"
                );
            }
        }
        
        # Return type object or instance
        if $return-type {
            return Type;
        } else {
            return Type.new(|%parameters);
        }
    }
    
    without $result { .throw }
    return $result;
}
```

### 2. Exports Introspection Methods

Add methods to introspect module exports, which is useful for discovery when modules export multiple types:

```raku
method module-exports(Str $module-name --> Map) {
    # Load the module
    my \M = (require ::($module-name));
    
    # Get exports if available
    if M.^can('EXPORT') {
        return M.EXPORT();
    }
    
    # Fallback: try to get exported symbols from package
    my %exports;
    try {
        my $package = ::($module-name);
        # Introspect package symbols
        # This is tricky in Raku - might need to use metamodel
        # For now, return empty map
    }
    
    return Map.new;
}

method exported-types(Str $module-name --> Array) {
    # Get all exported types from a module
    my @types;
    my %exports = self.module-exports($module-name);
    
    for %exports.kv -> $name, $symbol {
        # Check if symbol is a type
        if $symbol ~~ Mu && $symbol.^name ne 'Code' {
            try {
                # Try to get the type
                my \Type = ::($name);
                if Type.^name eq $name {
                    @types.push(Type);
                }
            }
        }
    }
    
    return @types;
}

method exported-types-doing(Str $module-name, Str $role-name --> Array) {
    # Find all exported types that do a specific role
    my @types = self.exported-types($module-name);
    my \Role = ::($role-name);
    
    return @types.grep(*.^does(Role)).Array;
}
```

### 3. Discovery Helper Methods

Add methods specifically for discovery patterns that combine module finding, loading, and role verification:

```raku
method discover-types-doing(
    :@paths = [],
    :@regexes = [],
    :@globs = [],
    Str :$does!,
    Bool :$load-modules = True
    --> Array
) {
    # Find all modules matching pattern
    my @all-mods = self.available-modules(@paths);
    
    # Filter modules by pattern
    my @candidate-modules;
    if @regexes.Bool {
        for @regexes -> $regex {
            @candidate-modules.push: |@all-mods.grep($regex);
        }
    } elsif @globs.Bool {
        for @globs -> $glob {
            my $match = Glob::Grammar.parse($glob, actions => Glob::ToRegexActions.new());
            my Str $pattern = $match.made;
            my $regex = qq|rx/$pattern/|.EVAL;
            @candidate-modules.push: |@all-mods.grep($regex);
        }
    } else {
        @candidate-modules = @all-mods;
    }
    
    my @found-types;
    my \Role = ::($does);
    
    for @candidate-modules.unique -> $module-name {
        try {
            if $load-modules {
                # Load module and check if main type does role
                my \M = (require ::($module-name));
                if M.^does(Role) {
                    @found-types.push(M);
                }
            }
            
            # Also check exported types
            my @exported = self.exported-types-doing($module-name, $does);
            @found-types.push: |@exported;
        }
    }
    
    return @found-types.unique.Array;
}
```

## Benefits Summary

### General Benefits

1. **Type Safety**: Verify role composition before use, catching errors early
2. **Flexibility**: Separate module names from type names, supporting various module structures
3. **Discovery Support**: Built-in methods for finding types that do specific roles
4. **Introspection**: Examine module exports to understand available types
5. **Performance**: Return type objects when instances aren't needed, avoiding unnecessary instantiation

## Implementation Considerations

### Backward Compatibility

The enhanced `load-library` should maintain backward compatibility:
- If only `:$type` is provided, behavior matches current implementation
- `:$module-name` and `:$does` are optional
- `:$return-type` defaults to `False` (current behavior)

### Error Handling

- Role verification failures should throw appropriate exceptions
- Module loading failures should be caught and reported
- Discovery methods should return empty arrays on failure, not throw

### Performance

- Module loading is cached via locks (existing behavior)
- Discovery can be expensive; consider caching results
- Type object returns avoid instantiation overhead during discovery


## Conclusion

The proposed enhancements to `Implementation::Loader` would:
1. Make it more generally useful for discovery patterns
3. Maintain backward compatibility
4. Provide type safety and better error handling

These enhancements are achievable and would significantly improve the usability of `Implementation::Loader` for general module discovery patterns.

